

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Serializable;
import java.nio.channels.FileChannel;
import java.util.Scanner;
import java.util.StringTokenizer;
import java.util.zip.GZIPInputStream;


/**
 * Very simple Dimacs file parser. Allow solvers to read the constraints from a
 * Dimacs formatted file.
 * 
 * It should be used that way:
 * 
 * <pre>
 * DimacsReader solver = new DimacsReader(SolverFactory.OneSolver());
 * solver.readInstance(&quot;mybench.cnf&quot;);
 * if (solver.isSatisfiable()) {
 * 	// SAT case
 * } else {
 * 	// UNSAT case
 * }
 * </pre>
 * 
 * That parser is not used for efficiency reasons. It will be updated with Java
 * 1.5 scanner feature.
 * 
 * @version 1.0
 * @author dlb
 * @author or
 */
public class DimacsReader implements Reader, Serializable {

	private static final long serialVersionUID = 1L;

	private int expectedNbOfClauses; // as announced on the p cnf line

	private final ISolver solver;

	public DimacsReader(ISolver solver) {
		this.solver = solver;
	}

	/**
	 * Skip comments at the beginning of the input stream.
	 * 
	 * @param in
	 *            the input stream
	 * @throws IOException
	 *             if an IO problem occurs.
	 */
	protected void skipComments(final LineNumberReader in) throws IOException {
		int c;

		do {
			in.mark(4);
			c = in.read();
			if (c == 'c') {
				in.readLine();
			} else {
				in.reset();
			}
		} while (c == 'c');
	}

	/**
	 * @param in
	 *            the input stream
	 * @throws IOException
	 *             iff an IO occurs
	 * @throws ParseFormatException
	 *             if the input stream does not comply with the DIMACS format.
	 */
	protected void readProblemLine(LineNumberReader in) throws IOException,
			ParseFormatException {

		String line = in.readLine();

		if (line == null) {
			throw new ParseFormatException(
					"premature end of file: <p cnf ...> expected  on line "
							+ in.getLineNumber());
		}
		StringTokenizer stk = new StringTokenizer(line);

		if (!(stk.hasMoreTokens() && stk.nextToken().equals("p")
				&& stk.hasMoreTokens() && stk.nextToken().equals("cnf"))) {
			throw new ParseFormatException(
					"problem line expected (p cnf ...) on line "
							+ in.getLineNumber());
		}

		int vars;

		// reads the max var id
		vars = Integer.parseInt(stk.nextToken());
		assert vars > 0;
		solver.newVar(vars);
		// reads the number of clauses
		expectedNbOfClauses = Integer.parseInt(stk.nextToken());
		assert expectedNbOfClauses > 0;

	}

	/**
	 * @param in
	 *            the input stream
	 * @throws IOException
	 *             iff an IO problems occurs
	 * @throws ParseFormatException
	 *             if the input stream does not comply with the DIMACS format.
	 * @throws ContradictionException
	 *             si le probl?me est trivialement inconsistant.
	 */
	// m?thode inutilis?e
	protected void readClauses(LineNumberReader in) throws IOException,
			ParseFormatException, ContradictionException {
		int lit;
		String line;
		Scanner scan;

		int realNbOfClauses = 0;

		IVecInt literals = new VecInt();

		while (true) {
			line = in.readLine();

			if (line == null) {
				// end of file
				if (literals.size() > 0) {
					// no 0 end the last clause
					solver.addClause(literals);
					realNbOfClauses++;
				}

				break;
			}

			if (line.startsWith("c ")) {
				// ignore comment line
				System.out.println("Found commmented line : " + line);
				continue;
			}
			if (line.startsWith("%")&&expectedNbOfClauses == realNbOfClauses) {
				System.out.println("Ignoring the rest of the file (SATLIB format");
				break;
			}
			scan = new Scanner(line);
			while (scan.hasNext()) {
				lit = scan.nextInt();

				if (lit != 0) {
					literals.push(lit);
				} else {
					if (literals.size() > 0) {
						solver.addClause(literals);
						literals.clear();
						realNbOfClauses++;
					}
				}

			}
		}

		if (expectedNbOfClauses != realNbOfClauses) {
			throw new ParseFormatException("wrong nbclauses parameter. Found "
					+ realNbOfClauses + ", " + expectedNbOfClauses
					+ " expected");
		}
	}

	/**
	 * @param in
	 *            the input stream
	 * @throws IOException
	 *             iff an IO problems occurs
	 * @throws ParseFormatException
	 *             if the input stream does not comply with the DIMACS format.
	 * @throws ContradictionException
	 *             si le probl?me est trivialement inconsistant.
	 */
	protected void readConstrs(LineNumberReader in) throws IOException,
			ParseFormatException, ContradictionException {
		int lit;
		String line;
		StringTokenizer stk;

		int realNbOfClauses = 0;

		IVecInt literals = new VecInt();

		while (true) {
			line = in.readLine();

			if (line == null) {
				// end of file
				if (literals.size() > 0) {
					// no 0 end the last clause
					solver.addClause(literals);
					realNbOfClauses++;
				}

				break;
			}

			if (line.startsWith("c ")) {
				// skip commented line
				continue;
			}
			if (line.startsWith("%")&&expectedNbOfClauses == realNbOfClauses) {
				System.out.println("Ignoring the rest of the file (SATLIB format");
				break;
			}
			stk = new StringTokenizer(line);
			String token;

			while (stk.hasMoreTokens()) {
				// on lit le prochain token
				token = stk.nextToken();

				if ((token.equals("<=")) || (token.equals(">="))) {
					// on est sur une contrainte de cardinalit?
					readCardinalityConstr(token, stk, literals);
					literals.clear();
					realNbOfClauses++;
				} else {
					lit = Integer.parseInt(token);

					if (lit != 0) {
						literals.push(lit);
					} else {
						if (literals.size() > 0) {
							solver.addClause(literals);
							literals.clear();
							realNbOfClauses++;
						}
					}

				}
			}
		}
		if (expectedNbOfClauses != realNbOfClauses) {
			throw new ParseFormatException("wrong nbclauses parameter. Found "
					+ realNbOfClauses + ", " + expectedNbOfClauses
					+ " expected");
		}
	}

	private void readCardinalityConstr(String token, StringTokenizer stk,
			IVecInt literals) throws ContradictionException,
			ParseFormatException {
		int card = Integer.parseInt(stk.nextToken());
		int lit = Integer.parseInt(stk.nextToken());
		if (lit == 0) {
			if (token.equals("<=")) {
				solver.addAtMost(literals, card);
			} else if (token.equals(">=")) {
				solver.addAtLeast(literals, card);
			}
		} else
			throw new ParseFormatException();
	}

	/**
	 * Remplit un prouveur ? partir d'un fichier Dimacs.
	 * 
	 * @param filename
	 *            le nom du fichier Dimacs (?ventuellement compress?)
	 * @throws FileNotFoundException
	 *             si le fichier n'est pas trouv?
	 * @throws ParseFormatException
	 *             si le fichier ne respecte pas le format Dimacs
	 * @throws IOException
	 *             pour un autre probl?me d'entr?e/sortie
	 * @throws ContradictionException
	 *             si le probl?me est trivialement inconsitant
	 */
	public IProblem parseInstance(String filename)
			throws FileNotFoundException, ParseFormatException, IOException,
			ContradictionException {

		FileInputStream fis = new FileInputStream(filename);
		InputStream input = fis;
		FileChannel channel = fis.getChannel();
		channel.map(FileChannel.MapMode.READ_ONLY, 0, channel.size());
		LineNumberReader reader;
		if (filename.endsWith(".gz")) {
			input = new GZIPInputStream(fis);
		}
		parseInstance(new LineNumberReader(new InputStreamReader(input)));
		return solver;
	}

	/**
	 * @param in
	 *            the input stream
	 * @throws ParseFormatException
	 *             if the input stream does not comply with the DIMACS format.
	 * @throws ContradictionException
	 *             si le probl?me est trivialement inconsitant
	 */
	public void parseInstance(LineNumberReader in) throws ParseFormatException,
			ContradictionException {
		solver.reset();
		try {
			skipComments(in);
			readProblemLine(in);
			readConstrs(in);
			// readClauses(in);
		} catch (IOException e) {
			throw new ParseFormatException(e);
		} catch (NumberFormatException e) {
			throw new ParseFormatException("integer value expected on line "
					+ in.getLineNumber(), e);
		}
	}

	public String decode(int[] model) {
		StringBuffer stb = new StringBuffer();
		for (int i = 0; i < model.length; i++) {
			stb.append(model[i]);
			stb.append(" ");
		}
		stb.append("0");
		return stb.toString();
	}
}
