

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.Scanner;
import java.util.zip.GZIPInputStream;



/**
 * This class is a quick hack to read opb formatted files.
 * 
 * The reader skip commented lines (beginning with COMMENT_SYMBOL)
 * and expect constraints of the form:
 * [name :] [[+|-]COEF] [+|-]LIT >=|<=|= DEGREE
 * where COEF and DEGREE are plain integer and LIT is an identifier.
 * 
 * @author leberre
 * 
 */
public class GoodOPBReader implements Reader,Serializable {

    /**
	 * Comment for <code>serialVersionUID</code>
	 */
	private static final long serialVersionUID = 1L;

	private static final String COMMENT_SYMBOL = "*";

    private ISolver solver;

    private final Map<String,Integer> map = new HashMap<String,Integer>();
    private final IVec<String> decode = new Vec<String>();
    
    /**
     *  
     */
    public GoodOPBReader(ISolver solver) {
        this.solver = solver;
    }

    public IProblem parseInstance(String filename) throws FileNotFoundException,
            ParseFormatException, IOException, ContradictionException {

        if (filename.endsWith(".gz")) {
            parseInstance(new LineNumberReader(new InputStreamReader(
                    new GZIPInputStream(new FileInputStream(filename)))));
        } else {
            parseInstance(new LineNumberReader(new FileReader(filename)));
        }
        return solver;
    }

    public void parseInstance(LineNumberReader in) throws ContradictionException, IOException {
        solver.reset();
        String line;
        while ((line = in.readLine()) != null) {
            // cannot trim is line is null
            line = line.trim();
            if (line.endsWith(";")) {
                line = line.substring(0,line.length()-1);
            }
            parseLine(line);
        }
    }

    void parseLine(String line) throws ContradictionException {
        // Skip commented line
        if (line.startsWith(COMMENT_SYMBOL))
            return;
        if (line.startsWith("p")) // ignore p cnf with pbchaff format
            return;
        if (line.startsWith("min:") || line.startsWith("min :"))
            return; // we will use that case later
        if (line.startsWith("max:") || line.startsWith("max :"))
            return; // we will use that case later

        // skip name of constraints:
        int index = line.indexOf(":");
        if (index!=-1) {
            line = line.substring(index+1);
        }
        
        IVecInt lits = new VecInt();
        IVecInt coeffs = new VecInt();
        Scanner stk = new Scanner(line);
        while (stk.hasNext()) {
            String token = stk.next();
            if (token.equals(">=") || token.equals("<=") || token.equals("=")) {
                assert stk.hasNext();
                int d = stk.nextInt();
                if (token.equals(">=") || token.equals("=")) {
                    solver.addPseudoBoolean(lits, coeffs, true, d);
                }
                if (token.equals("<=") || token.equals("=")) {
                    solver.addPseudoBoolean(lits, coeffs, false, d);
                }
            } else {
                if (token.equals("+")) {
                    assert stk.hasNext();
                    token = stk.next();
                } else if (token.equals("-")) {
                    assert stk.hasNext();
                    token = token+stk.next();
                }
                int coef;
                // should contain a coef and a literal
                try {
                    // we need to remove + from the integer
                    if (token.startsWith("+")) {
                        token = token.substring(1);
                    }
                    coef = Integer.parseInt(token);
                    assert stk.hasNext();
                    token = stk.next();
                } catch (NumberFormatException nfe) {
                    // its only an identifier
                    coef = 1;
                }
                if (token.equals("+")||token.equals("-")||token.equals("~")) {
                    assert stk.hasNext();
                    token = token+stk.next();
                }
                boolean negative = false;
                if (token.startsWith("+")) {
                    token = token.substring(1);
                } else if (token.startsWith("-")) {
                    token = token.substring(1);
                    assert coef ==1;
                    coef = -1;
                } else if (token.startsWith("~")) {
                    token = token.substring(1);
                    negative = true;
                }

                Integer id = map.get(token);
                if (id == null) {
                    map.put(token, id = solver.newVar());
                    decode.push(token);
                    assert decode.size()==id.intValue();
                }
                coeffs.push(coef);
                int lid = (negative?-1:1)*id.intValue();
                lits.push(lid);
                assert coeffs.size()==lits.size();
            }
        }
    }
    
    public String decode(int [] model) {
        StringBuffer stb = new StringBuffer();
        for (int i=0;i<model.length;i++) {
            if (model[i]<0) {
                stb.append("-");
                stb.append(decode.get(-model[i]-1));
            } else {
                stb.append(decode.get(model[i]-1));
            }
            stb.append(" ");
        }
        return stb.toString();
    }

}