

import java.io.PrintStream;




/**
 * @author leberre
 * 
 * That interface contains all the services available on a SAT solver.
 *  
 */
public interface ISolver extends IProblem {

    /**
     * Create a new variable in the solver (and thus in the vocabulary).
     * 
     * @return the number of variables available in the vocabulary, which is
     *         the identifier of the new variable.
     */
    int newVar();

    /**
     * Create <code>howmany</code> variables in the solver (and thus in the
     * vocabulary).
     * 
     * @param howmany
     *            number of variables to create
     * @return the total number of variables available in the solver (the
     *         highest variable number)
     */
    int newVar(int howmany);

    /**
     * Create a clause from a set of literals The literals are represented by
     * non null integers such that opposite literals a represented by opposite
     * values. (clasical Dimacs way of representing literals).
     * 
     * @param literals
     *            a set of literals
     * @return false iff a trivial inconsistency is detected.
     */
    void addClause(IVecInt literals) throws ContradictionException;

    /**
     * Create clauses from a set of set of literals. This is convenient to create
     * in a single call all the clauses (mandatory for the distributed version of 
     * the solver).
     * It is mainly a loop to addClause().
     * 
     * @param clauses a vector of set (VecInt) of literals in the dimacs format. 
     * @throws ContradictionException iff a trivial inconsistency is detected.
     * @see #addClause(VecInt)
     */
    void addAllClauses(IVec clauses) throws ContradictionException;
    	
    /**
     * Create a cardinality constraint of the type "at most n of those literals
     * must be satisfied"
     * 
     * @param literals
     *            a set of literals
     * @param degree
     *            the degree of the cardinality constraint
     * 
     * @return false iff a trivial inconsistency is detected.
     */

    void addAtMost(IVecInt literals, int degree) throws ContradictionException;

    /**
     * Create a cardinality constraint of the type "at least n of those
     * literals must be satisfied"
     * 
     * @param literals
     *            a set of literals
     * @param degree
     *            the degree of the cardinality constraint
     * 
     * @return false iff a trivial inconsistency is detected.
     */
    void addAtLeast(IVecInt literals, int degree) throws ContradictionException;

    /**
     * Create a Pseudo-Boolean constraint of the type "at least n of those
     * literals must be satisfied"
     * 
     * @param literals
     *            a set of literals
     * @param coeffs
     * 			the coefficients of the literals 
     * @param n
     *            the degree of the cardinality constraint
     * @param moreThan
     * 			true if it is a constraint >= degree  
     * @return false iff a trivial inconsistency is detected.
     */
    void addPseudoBoolean(IVecInt literals, IVecInt coeffs, boolean moreThan, int degree) throws ContradictionException;

    /**
     * To set the internal timeout of the solver. When the timeout is reached,
     * a timeout exception is launched by the solver.
     * 
     * @param t
     *            the timeout (in s)
     */
    void setTimeout(int t);

    /**
     * Clean up the internal state of the solver.
     */
    void reset();

    /**
     * AVOID. DOES NOT WORK YET. to back-up number of constraints and number of
     * learned constraints when computing minimal model sets.
     *  
     */
    public void backUp();

    /**
     * AVOID. DOES NOT WORK YET. to restore number of constraints and number of
     * learned constraints when computing minimal model sets.
     *  
     */
    public void restore();

	/**
	 * Display statistics to the given output stream
	 *  
	 * @param out 
	 * @param prefix the prefix to put in front of each line
	 */
	void printStat(PrintStream out, String prefix);

	/**
	 * Display a textual representation of the solver configuration.
	 *  
	 * @param prefix the prefix to use on each line.
	 * @return
	 */
	String toString(String prefix);
}