/*
 * SAT4J: a SATisfiability library for Java   
 * Copyright (C) 2004 Daniel Le Berre
 * 
 * Based on the original minisat specification from:
 * 
 * An extensible SAT solver. Niklas E?n and Niklas S?rensson.
 * Proceedings of the Sixth International Conference on Theory 
 * and Applications of Satisfiability Testing, LNCS 2919, 
 * pp 502-518, 2003.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *  
 */



import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;

/*
 * Created on 11 nov. 2003
 * 
 * To change the template for this generated file go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */

/**
 * @author leberre
 * 
 * To change the template for this generated type comment go to
 * Window&gt;Preferences&gt;Java&gt;Code Generation&gt;Code and Comments
 */
public final class Lanceur {

	private static final int SAT_EXIT_CODE = 10;
	private static final int UNSAT_EXIT_CODE = 20;
	private static final int UNKNOWN_EXIT_CODE = 0;
	
	private Lanceur() {
	}

	

	/**
	 * Lance le prouveur sur un fichier Dimacs
	 * 
	 * @param args
	 *            doit contenir le nom d'un fichier Dimacs, eventuellement
	 *            compress?.
	 */
	public static void main(String[] args) {
		try {
			displayHeader();

			LimitedLearning learning = new LimitedLearning(10);
			Solver solver = new Solver(new FirstUIP(), learning, new MixedDataStructureWithBinary());
			learning.setSolver(solver);



			long begintime = System.currentTimeMillis();

			readProblem(args, solver, begintime);

			boolean isSat = solver.isSatisfiable();

			displayResult(solver, begintime, isSat);

		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ContradictionException e) {
			System.out.println("s UNSATISFIABLE");
			System.out.println("c (trivial)");
			System.exit(UNSAT_EXIT_CODE);
		} catch (TimeoutException e) {
			System.out.println("c timeout");
		} catch (ParseFormatException e) {
			e.printStackTrace();
		}
		System.exit(UNKNOWN_EXIT_CODE);
	}

	/**
	 * @param solver
	 * @param begintime
	 * @param isSat
	 */
	private static void displayResult(ISolver solver, long begintime,
			boolean isSat) {
		solver.printStat(System.out, "c ");
		System.out.println("c Total CPU time (ms) : "
				+ (System.currentTimeMillis() - begintime) / 1000.0);
		if (isSat) {
			System.out.println("s SATISFIABLE");
			int[] model = solver.model();
			System.out.println("v " + reader.decode(model));
			System.exit(SAT_EXIT_CODE);
		} else {
			System.out.println("s UNSATISFIABLE");
			System.exit(UNSAT_EXIT_CODE);
		}
	}

	/**
	 * @param args
	 * @param solver
	 * @param begintime
	 * @throws FileNotFoundException
	 * @throws ParseFormatException
	 * @throws IOException
	 * @throws ContradictionException
	 */
	private static void readProblem(String[] args, ISolver solver,
			long begintime) throws FileNotFoundException, ParseFormatException,
			IOException, ContradictionException {
		System.out.println("c solving " + args[0]);

		System.out.print("c reading problem ");
		reader = new InstanceReader(solver);
		reader.parseInstance(args[0]);
		System.out.println("time " + (System.currentTimeMillis() - begintime)
				/ 1000.0);
		System.out.println("c #vars     " + solver.nVars());
		System.out.println("c #clauses  " + solver.nConstraints());
	}



	/**
	 * @throws IOException
	 */
	private static void displayHeader() throws IOException {
		System.out
				.println("c SAT4J: a SATisfiability library for Java (c) 2004-2005 Daniel Le Berre");
		System.out
				.println("c This is free software under the GNU LGPL licence. See www.sat4j.org for details.");
		URL url = Lanceur.class.getResource("/sat4j.version");
		if (url != null) {
			BufferedReader in = new BufferedReader(new InputStreamReader(url
					.openStream()));
			System.out.println("c version " + in.readLine());
			in.close();
		} else {
			System.out.println("c no version file found!!!");
		}
	}

	private static InstanceReader reader;
}
