

import java.io.PrintStream;
import java.io.Serializable;
import java.util.Comparator;
import java.util.Timer;
import java.util.TimerTask;

/**
 * @author leberre
 * 
 */
public class Solver implements ISolver, UnitPropagationListener,
		ActivityListener, Learner, Serializable {

	private static final long serialVersionUID = 1L;

	private static final double CLAUSE_RESCALE_FACTOR = 1e-20;

	private static final double CLAUSE_RESCALE_BOUND = 1 / CLAUSE_RESCALE_FACTOR;

	/**
	 * List des contraintes du probl?me.
	 */
	private final IVec<Constr> constrs = new Vec<Constr>(); // Constr

	// vector

	/**
	 * Liste des clauses apprises.
	 */
	private final IVec<Constr> learnts = new Vec<Constr>(); // Clause

	// vector

	/**
	 * incr?ment pour l'activit? des clauses.
	 */
	private double claInc = 1.0;

	/**
	 * decay factor pour l'activit? des clauses.
	 */
	private double claDecay = 1.0;

	/**
	 * Queue de propagation
	 */
	private final IntQueue propQ = new IntQueue(); // Lit

	// queue

	/**
	 * affectation en ordre chronologique
	 */
	protected final IVecInt trail = new VecInt(); // lit

	// vector

	/**
	 * indice des s?parateurs des diff?rents niveau de d?cision dans trail
	 */
	private final IVecInt trailLim = new VecInt(); // int

	// vector

	/**
	 * S?pare les hypoth?ses incr?mentale et recherche
	 */
	private int rootLevel;

	private int[] model = null;

	protected final ILits voc;

	private VarOrder order;

	private final Comparator<Constr> comparator = new ActivityComparator();

	private final SolverStats stats = new SolverStats();

	private final LearningStrategy learner;

	protected final AssertingClauseGenerator analyzer;

	private boolean undertimeout;

	private int timeout = Integer.MAX_VALUE;

	protected final DataStructureFactory dsfactory;

	private final SearchParams params;

	private final IVecInt __dimacs_out = new VecInt();

	private IVecInt dimacs2internal(IVecInt in) {
		__dimacs_out.clear();
		__dimacs_out.ensure(in.size());
		for (int i = 0; i < in.size(); i++) {
			assert (in.get(i) != 0) && (Math.abs(in.get(i)) <= voc.nVars());
			__dimacs_out.unsafePush(voc.getFromPool(in.get(i)));
		}
		return __dimacs_out;
	}

	public Solver(AssertingClauseGenerator acg, LearningStrategy learner,
			DataStructureFactory dsf) {
		
		this(acg, learner, dsf, new SearchParams(), new VarOrder());
		
	}

	/**
	 * creates a Solver without LearningListener.
	 * 
	 * A learningListener must be added to the solver, else it won't
	 * backtrack!!! A data structure factory must be provided, else it won't
	 * work either.
	 * 
	 * @param acg
	 *            an asserting clause generator
	 */

	public Solver(AssertingClauseGenerator acg, LearningStrategy learner,
			DataStructureFactory dsf, VarOrder order) {
		this(acg, learner, dsf, new SearchParams(), order);
		
	}

	public Solver(AssertingClauseGenerator acg, LearningStrategy learner,
			DataStructureFactory dsf, SearchParams params) {
		this(acg, learner, dsf, params, new VarOrder());
		
	}

	public Solver(AssertingClauseGenerator acg, LearningStrategy learner,
			DataStructureFactory dsf, SearchParams params, VarOrder order) {
		analyzer = acg;
		this.learner = learner;
		dsfactory = dsf;
		dsfactory.setUnitPropagationListener(this);
		dsfactory.setLearner(this);
		this.order = order;
		voc = dsf.getVocabulary();
		order.setLits(voc);
		this.params = params;
		
	}

	public void setTimeout(int t) {
		timeout = t;
	}

	protected int nAssigns() {
		return trail.size();
	}

	public int nConstraints() {
		return constrs.size();
	}

	private int nLearnts() {
		return learnts.size();
	}

	public void learn(Constr c) {
		learnts.push(c);
		c.setLearnt();
		c.register();
		claBumpActivity(c);
		stats.learnedclauses++;
	}

	public int decisionLevel() {
		return trailLim.size();
	}

	public int newVar() {
		int index = voc.nVars() + 1;
		voc.ensurePool(index);
		seen = new boolean[index + 1];
		trail.ensure(index);
		trailLim.ensure(index);
		propQ.ensure(index);
		order.newVar();
		return index;
	}

	public int newVar(int howmany) {
		voc.ensurePool(howmany);
		order.newVar(howmany);
		seen = new boolean[howmany + 1];
		trail.ensure(howmany);
		trailLim.ensure(howmany);
		propQ.ensure(howmany);
		return voc.nVars();
	}

	public void addClause(IVecInt literals) throws ContradictionException {
		IVecInt vlits = dimacs2internal(literals);
		addConstr(dsfactory.createClause(vlits));
		// VecInt vlits = dimacs2internal(literals);
		// VecInt coefs = new VecInt();
		// coefs.growTo(vlits.size(),1);
		// Constr c = PbContr.pbContrNew(this,lits,vlits,coefs,true,1);
		// if (c != null) {
		// constrs.push(c);
		// assert c instanceof PbContr;
		// }
		// System.out.println(" Adding "+c);
		// for (int i=0;i<literals.size();i++) {
		// literals.set(i,-literals.get(i));
		// }
		// addAtMost(literals,literals.size()-1);
		// addAtLeast(literals,1);
	}

	public void addPseudoBoolean(IVecInt literals, IVecInt coeffs,
			boolean moreThan, int degree) throws ContradictionException {
		IVecInt vlits = dimacs2internal(literals);
		assert vlits.size() == literals.size();
		assert literals.size() == coeffs.size();
		addConstr(dsfactory.createPseudoBooleanConstraint(vlits, coeffs,
				moreThan, degree));
	}

	public void addAllClauses(IVec clauses) throws ContradictionException {
		for (int i = 0; i < clauses.size(); i++) {
			addClause((IVecInt) clauses.get(i));
		}
	}

	public void addAtMost(IVecInt literals, int degree)
			throws ContradictionException {
		for (int i = 0; i < literals.size(); i++) {
			literals.set(i, -literals.get(i));
		}
		addAtLeast(literals, literals.size() - degree);
	}

	public void addAtLeast(IVecInt literals, int degree)
			throws ContradictionException {
		IVecInt vlits = dimacs2internal(literals);
		addConstr(dsfactory.createCardinalityConstraint(vlits, degree));
	}

	public boolean simplifyDB() {
		// aucune raison de recommencer un propagate?
		// if (propagate() != null) {
		// // Un conflit est d?couvert, la base est inconsistante
		// return false;
		// }

		// Simplifie la base de clauses apres la premiere propagation des
		// clauses unitaires
		IVec<Constr>[] cs = new IVec[] { constrs, learnts };
		for (int type = 0; type < 2; type++) {
			int j = 0;
			for (int i = 0; i < cs[type].size(); i++) {
				if (cs[type].get(i).simplify()) {
					// enleve les contraintes satisfaites de la base
					cs[type].get(i).remove();
				} else {
					cs[type].set(j++, cs[type].get(i));
				}
			}
			cs[type].shrinkTo(j);
		}
		return true;
	}

	/**
	 * Si un mod?le est trouv?, ce vecteur contient le mod?le.
	 * 
	 * @return un mod?le de la formule.
	 */
	public int[] model() {
		if (model == null) {
			throw new UnsupportedOperationException(
					"Call the solve method first!!!");
		}
		int[] nmodel = new int[model.length];
		System.arraycopy(model, 0, nmodel, 0, model.length);
		return nmodel;
	}

	/**
	 * Satisfait un litt?ral
	 * 
	 * @param p
	 *            le litt?ral
	 * @return true si tout se passe bien, false si un conflit appara?t.
	 */
	public boolean enqueue(int p) {
		return enqueue(p, null);
	}

	/**
	 * Satisfait un litt?ral
	 * 
	 * @param p
	 *            le litt?ral
	 * @param from
	 *            la raison de satisfaire le litt?ral
	 * @return true si tout se passe bien, false si un conflit appara?t.
	 */
	public boolean enqueue(int p, Constr from) {
		// System.err.println("prop "+Lits.toString(p)+" thanks to "+from);
		assert p > 1;
		if (!voc.isUnassigned(p)) {
			if (voc.isFalsified(p)) {
				// conflicting enqueued assignment
				return false;
			}
			// existing consistent assignment, do not enqueue
			return true;
		}
		// new fact, store it
		voc.satisfies(p);
		voc.setLevel(p, decisionLevel());
		voc.setReason(p, from);
		trail.push(p);
		propQ.insert(p);
		return true;
	}

	private boolean[] seen = new boolean[0];

	private final IVecInt preason = new VecInt();

	private final IVecInt outLearnt = new VecInt();

	public int analyze(Constr confl, Handle outLearntRef) {
		assert confl != null;
		outLearnt.clear();

		assert outLearnt.size() == 0;
		for (int i = 0; i < seen.length; i++) {
			seen[i] = false;
		}

		analyzer.initAnalyze();
		int p = ILits.UNDEFINED;

		outLearnt.push(ILits.UNDEFINED);
		// reserve de la place pour le litteral falsifie
		int outBtlevel = 0;

		do {
			preason.clear();
			assert confl != null;
			confl.calcReason(p, preason);
			if (confl.learnt()) {
				claBumpActivity(confl);
			}
			// Trace reason for p
			for (int j = 0; j < preason.size(); j++) {
				int q = preason.get(j);
				if (!seen[q >> 1]) {
					seen[q >> 1] = true;
					if (voc.getLevel(q) == decisionLevel()) {
						analyzer.onCurrentDecisionLevelLiteral(q);
					} else if (voc.getLevel(q) > 0) {
						// ajoute les variables depuis le niveau de d?cision 0
						outLearnt.push(q ^ 1);
						outBtlevel = Math.max(outBtlevel, voc.getLevel(q));
					}
				}
			}

			// select next reason to look at
			do {
				p = trail.last();
				// System.err.print((Clause.lastid()+1)+"
				// "+((Clause)confl).getId()+" ");
				confl = voc.getReason(p);
				// System.err.println(((Clause)confl).getId());
				// assert(confl != null) || counter == 1;
				undoOne();
			} while (!seen[p >> 1]);
			// seen[p.var] indique que p se trouve dans outLearnt ou dans
			// le dernier niveau de d?cision
		} while (analyzer.clauseNonAssertive(confl));

		outLearnt.set(0, p ^ 1);

		Constr c = dsfactory.createUnregisteredClause(outLearnt);

		outLearntRef.obj = c;

		assert outBtlevel > -1;
		return outBtlevel;
	}

	/**
	 * 
	 */
	protected void undoOne() {
		// recupere le dernier litteral affecte
		int p = trail.last();
		assert p > 1;
		int x = p >> 1;
		// desaffecte la variable
		voc.unassign(p);
		voc.setReason(p, null);
		voc.setLevel(p, -1);
		// met a jour l'heuristique
		order.undo(x);
		// depile le litteral des affectations
		trail.pop();
		// met a jour les contraintes apres desaffectation du litteral :
		// normalement, il n'y a rien a faire ici pour les prouveurs de type
		// Chaff??
		IVec undos = voc.undos(p);
		assert undos != null;
		while (undos.size() > 0) {
			((Undoable) undos.last()).undo(p);
			undos.pop();
		}
	}

	/**
	 * @param outclause
	 */
	public void claBumpActivity(Constr confl) {
		confl.incActivity(claInc);
		if (confl.getActivity() > CLAUSE_RESCALE_BOUND)
			claRescalActivity();
		for (int i = 0; i < confl.size(); i++) {
			varBumpActivity(confl.get(i));
		}
	}

	/**
	 * @param object
	 */
	public void varBumpActivity(int p) {
		order.updateVar(p);
	}

	private void claRescalActivity() {
		for (int i = 0; i < learnts.size(); i++) {
			learnts.get(i).rescaleBy(CLAUSE_RESCALE_FACTOR);
		}
		claInc *= CLAUSE_RESCALE_FACTOR;
	}

	/**
	 * @return null if not conflict is found, else a conflicting constraint.
	 */
	public Constr propagate() {
		while (propQ.size() > 0) {
			stats.propagations++;
			int p = propQ.dequeue();
			// p est maintenant le litt?ral a propager
			// TODO Move that part to Constr to avoid
			// watches manipulation in counter Based clauses for instance.
			assert p > 1;
			IVec<Propagatable> constrs = dsfactory.getWatchesFor(p);
			// 
			for (int i = 0; i < constrs.size(); i++) {
				stats.inspects++;
				if (!constrs.get(i).propagate(this, p)) {
					// Constraint is conflicting: copy remaining watches to
					// watches[p]
					// and return constraint
					dsfactory.conflictDetectedInWatchesFor(p, i);
					propQ.clear();
					// FIXME enlever le transtypage
					return (Constr) constrs.get(i);
				}
			}
		}
		return null;
	}

	void record(Constr constr) {
		constr.setLearnt();
		constr.assertConstraint(this);
		if (constr.size() == 1) {
			stats.learnedliterals++;
		} else {
			learner.learns(constr);
		}
	}

	/**
	 * @return false ssi conflit imm?diat.
	 */
	public boolean assume(int p) {
		// Precondition: assume propagation queue is empty
		assert propQ.size() == 0;
		trailLim.push(trail.size());
		return enqueue(p);
	}

	/**
	 * Revert to the state before the last push()
	 * 
	 */
	private void cancel() {
		assert propQ.size() == 0 || !undertimeout;

		for (int c = trail.size() - trailLim.last(); c > 0; c--) {
			undoOne();
		}
		trailLim.pop();
	}

	/**
	 * Cancel several levels of assumptions
	 * 
	 * @param level
	 */
	protected void cancelUntil(int level) {
		while (decisionLevel() > level) {
			cancel();
		}
	}

	private final Handle learntConstraint = new Handle();

	Lbool search(long nofConflicts, long nofLearnts) {
		assert rootLevel == decisionLevel();
		stats.starts++;
		int conflictC = 0;

		// varDecay = 1 / params.varDecay;
		order.setVarDecay(1 / params.getVarDecay());
		claDecay = 1 / params.getClaDecay();

		do {
			// propage les clauses unitaires
			Constr confl = propagate();
			assert propQ.size() == 0;

			if (confl != null) {
				// un conflit apparait
				stats.conflicts++;
				conflictC++;
				if (decisionLevel() == rootLevel) {
					// on est a la racine, la formule est inconsistante
					return Lbool.FALSE;
				}

				// analyse la cause du conflit
				assert confl != null;
				int backtrackLevel = analyze(confl, learntConstraint);
				assert backtrackLevel < decisionLevel();
				cancelUntil(Math.max(backtrackLevel, rootLevel));
				assert (decisionLevel() >= rootLevel)
						&& (decisionLevel() >= backtrackLevel);
				if (learntConstraint.obj == null) {
					return Lbool.FALSE;
				}
				record((Constr) learntConstraint.obj);
				learntConstraint.obj = null;
				decayActivities();
			} else {
				// No conflict found
				if (decisionLevel() == 0) {
					// Simplify the set of problem clause
					// iff rootLevel==0
					stats.rootSimplifications++;
					boolean ret = simplifyDB();
					assert ret;
				}
				// was learnts.size() - nAssigns() > nofLearnts
				if (nofLearnts >= 0 && learnts.size() > nofLearnts) {
					// Reduce the set of learnt clauses
					reduceDB();
				}
				assert nAssigns() <= voc.nVars();
				if (nAssigns() == voc.nVars()) {
					modelFound();
					return Lbool.TRUE;
				}
				if (conflictC >= nofConflicts) {
					// Reached bound on number of conflicts
					// Force a restart
					cancelUntil(rootLevel);
					return Lbool.UNDEFINED;
				}
				// New variable decision
				stats.decisions++;
				int p = order.select();
				assert p > 1;
				boolean ret = assume(p);
				assert ret;
			}
		} while (undertimeout);
		return Lbool.UNDEFINED; // timeout occured
	}

	/**
	 * 
	 */
	void modelFound() {
		// Model found
		model = new int[trail.size()];
		int index = 0;
		for (int i = 1; i <= voc.nVars(); i++) {
			if (!voc.isUnassigned(i)) {
				model[index++] = voc.isSatisfied(voc.getFromPool(i)) ? i : -i;
			}
		}
		assert index == model.length;
		cancelUntil(rootLevel);
	}

	/**
	 * 
	 */
	protected void reduceDB() {
		int i, j;
		double lim = claInc / learnts.size();
		sortOnActivity();
		for (i = j = 0; i < learnts.size() / 2; i++) {
			Constr c = learnts.get(i);
			if (!c.locked()) {
				c.remove();
			} else {
				learnts.set(j++, learnts.get(i));
			}
		}
		for (; i < learnts.size(); i++) {
			Constr c = learnts.get(i);
			if (!c.locked() && (c.getActivity() < lim)) {
				c.remove();
			} else {
				learnts.set(j++, learnts.get(i));
			}
		}
		learnts.shrinkTo(j);
	}

	/**
	 * @param learnts
	 */
	private void sortOnActivity() {
		learnts.sort(comparator);
	}

	/**
	 * 
	 */
	protected void decayActivities() {
		order.varDecayActivity();
		claDecayActivity();
	}

	/**
	 * 
	 */
	private void claDecayActivity() {
		claInc *= claDecay;
	}

	/**
	 * @return
	 */
	public boolean isSatisfiable() throws TimeoutException {
		return isSatisfiable(new VecInt());
	}

	public boolean isSatisfiable(IVecInt assumps) throws TimeoutException {
		Lbool status = Lbool.UNDEFINED;
		double nofConflicts = params.initConflictBound;
		double nofLearnts = nConstraints()
				* params.initLearntBoundConstraintFactor;

		stats.reset();
		order.init();
		model = null; // forget about previous model

		// propagate constraints
		if (propagate() != null) {
			cancelUntil(0);
			return false;
		}

		// push incremental assumptions
		for (int i = 0; i < assumps.size(); i++) {
			if (!assume(voc.getFromPool(assumps.get(i)))
					|| (propagate() != null)) {
				cancelUntil(0);
				return false;
			}
		}
		// StringBuffer stb = new StringBuffer();
		// stb.append("%RESA ");
		// stb.append(nVars());
		// stb.append(" ");
		// stb.append(nConstraints());
		// while(stb.length()<255) {
		// stb.append(' ');
		// }
		// System.err.println(stb);
		rootLevel = decisionLevel();

		TimerTask stopMe = new TimerTask() {
			public void run() {
				undertimeout = false;
			}
		};
		undertimeout = true;
		Timer timer = new Timer(true);
		timer.schedule(stopMe, timeout * 1000L);

		// Solve
		while ((status == Lbool.UNDEFINED) && undertimeout) {
			status = search(Math.round(nofConflicts), Math.round(nofLearnts));
			nofConflicts *= params.conflictBoundIncFactor;
			nofLearnts *= params.learntBoundIncFactor;
		}

		cancelUntil(0);
		timer.cancel();
		if (!undertimeout) {
			throw new TimeoutException(" Timeout (" + timeout + "s) exceeded");
		}
		return status == Lbool.TRUE;
	}

	public SolverStats getStats() {
		return stats;
	}

	public VarOrder getOrder() {
		return order;
	}

	public void setOrder(VarOrder h) {
		order = h;
		order.setLits(voc);
	}

	public ILits getVocabulary() {
		return voc;
	}

	public void reset() {
		voc.resetPool();
		dsfactory.reset();
	}

	public int nVars() {
		return voc.nVars();
	}

	private int oldNConstraints;

	private int oldNLearnts;

	private int oldTrail;

	private int oldTrailLim;

	/**
	 * to back-up number of constraints and number of learned constraints when
	 * computing minimal model sets.
	 * 
	 */
	public void backUp() {
		assert propQ.size() == 0;
		assert decisionLevel() == rootLevel;
		oldNConstraints = nConstraints();
		oldNLearnts = nLearnts();
		oldTrail = trail.size();
		oldTrailLim = trailLim.size();
	}

	/**
	 * to restore number of constraints and number of learned constraints when
	 * computing minimal model sets.
	 * 
	 */
	public void restore() {
		assert oldTrailLim == trailLim.size();
		assert oldTrail <= trail.size();
		while (nConstraints() > oldNConstraints) {
			constrs.last().remove();
			constrs.pop();
		}
		while (nLearnts() > oldNLearnts) {
			learnts.last().remove();
			learnts.pop();
		}

		while (trail.size() > oldTrail)
			undoOne();

		oldNConstraints = -1;
		oldNLearnts = -1;
		oldTrail = oldTrailLim = -1;
	}

	/**
	 * @param constr
	 *            a constraint implementing the Constr interface.
	 */
	public void addConstr(Constr constr) {
		if (constr != null) {
			constrs.push(constr);
		}

	}

	public DataStructureFactory getDSFactory() {
		return dsfactory;
	}

	public IVecInt getOutLearnt() {
		return outLearnt;
	}

	/**
	 * returns the ith constraint in the solver.
	 * 
	 * @param i
	 *            the constraint number (begins at 0)
	 * @return the ith constraint
	 */
	public Constr getIthConstr(int i) {
		return constrs.get(i);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.sat4j.specs.ISolver#printStat(java.io.PrintStream,
	 *      java.lang.String)
	 */
	public void printStat(PrintStream out, String prefix) {
		out.println(prefix + "starts\t: " + stats.starts);
		out.println(prefix + "conflicts\t: " + stats.conflicts);
		out.println(prefix + "decisions\t: " + stats.decisions);
		out.println(prefix + "propagations\t: " + stats.propagations);
		out.println(prefix + "inspects\t: " + stats.inspects);
		out.println(prefix + "learned literals\t: " + stats.learnedliterals);
		out.println(prefix + "learned binary clauses\t: "
				+ stats.learnedbinaryclauses);
		out.println(prefix + "learned ternary clauses\t: "
				+ stats.learnedternaryclauses);
		out.println(prefix + "learned clauses\t: " + stats.learnedclauses);
		out.println(prefix + "root simplifications\t: "
				+ stats.rootSimplifications);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString(String prefix) {
		StringBuilder stb = new StringBuilder();
		Object[] objs = { analyzer, dsfactory, learner, params };
		stb.append(prefix);
		stb.append("--- Begin Solver configuration ---");
		stb.append("\n");
		for (Object o : objs) {
			stb.append(prefix);
			stb.append(o.toString());
			// FIXME
			stb.append("\n");
		}
		stb.append(prefix);
		stb.append("--- End Solver configuration ---");
		return stb.toString();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		return toString("");
	}
}

class ActivityComparator implements Comparator<Constr>, Serializable {

	private static final long serialVersionUID = 1L;

	/*
	 * (non-Javadoc)
	 * 
	 * @see java.util.Comparator#compare(java.lang.Object, java.lang.Object)
	 */
	public int compare(Constr c1, Constr c2) {
		return (int) Math.round(c1.getActivity() - c2.getActivity());
	}
}